"""
Quest Screenshot Registry
========================
Maps quest IDs → quest-level images and objective-specific images.
All URLs use CDN. Fields like normalizedName, type, description are
for human readability only — not used by code.
"""

CDN_BASE_URL = "https://cdn.tarkovquestie.com/images/quests"


def _img(quest_name: str, filename: str, text: str = "") -> dict:
    """Build CDN image dict."""
    return {
        "url": f"{CDN_BASE_URL}/{quest_name}/{filename}",
        "text": text or None,
    }


# Nested registry: quest_id → { normalizedName, images, objectives }
# objectives: objective_id → { type, description, images }
#
# Lookup priority:
#   1. Objective-specific images (if present)
#   2. Quest-level images (fallback for any objective)
#   3. Empty list (no images in registry)
QUEST_SCREENSHOTS = {
    "63a88045abf76d719f42d715": {
        "normalizedName": "the-delicious-sausage",
        "images": [],
        "objectives": {
            "63a98cfbc31b00242d28a95b": {
                "type": "visit",
                "description": "Scout the Shestyorochka store at Nikitskaya street",
                "images": [
                    _img("the-delicious-sausage", "the_shestyorochka_store.png", "The Shestyorochka store"),
                ],
            },
            "63a98d24655ec5555b4aa9e7": {
                "type": "visit",
                "description": "Scout the Sparja store at Primorsky ave",
                "images": [
                    _img("the-delicious-sausage", "the_primorsky_ave_sparja_store.png", "The Primorsky ave Sparja store"),
                ],
            },
            "63a98d39da7999196148ba3a": {
                "type": "visit",
                "description": "Scout the Sparja store in Pinewood hotel",
                "images": [
                    _img("the-delicious-sausage", "the_pinewood_hotel_sparja_store.png", "The Pinewood hotel Sparja store"),
                ],
            },
            "63a98d60c0f61a5d8731cd9f": {
                "type": "visit",
                "description": "Scout the Goshan store in Concordia",
                "images": [
                    _img("the-delicious-sausage", "outside_concordia_house_no._3_goshan_store_marked.png", "Outside Concordia house no. 3"),
                    _img("the-delicious-sausage", "inside_concordia_house_no._3_door_to_the_goshan_store.png", "Inside Concordia house no. 3, door to the Goshan store"),
                ],
            },
        },
    },
    "66058cbd9f59e625462acc8e": {
        "normalizedName": "create-a-distraction-part-1-pvp-zone",
        "images": [],
        "objectives": {
            "660a9dc64c0927ead4fcfeef": {
                "type": "plantItem",
                "description": "Plant TP-200 TNT bricks on the second floor of the cafe on Ground Zero",
                "images": [
                    _img("create-a-distraction-part-1-pvp-zone", "the_olive_restaurant.png", "The Olive restaurant"),
                    _img("create-a-distraction-part-1-pvp-zone", "stashing_location_on_the_second_floor.png", "Stashing location on the second floor"),
                ],
            },
        },
    },
    "639135d89444fb141f4e6eea": {
        "normalizedName": "population-census",
        "images": [],
        "objectives": {
            "639135d89444fb141f4e6eeb": {
                "type": "findQuestItem",
                "description": "Obtain the journal containing resident details on Streets of Tarkov",
                "images": [
                    _img("population-census", "housing_department_building_backside_with_entrance.png", "Housing department building backside with entrance"),
                    _img("population-census", "hall1.png", "Hall1"),
                    _img("population-census", "spawn1.png", "Spawn1"),
                    _img("population-census", "populationcensusthirdofficerightlampdesk.png", "Third Office Right LampDesk"),
                    _img("population-census", "populationcensus3spawn.png", "Populationcensus 3 spawn"),
                    _img("population-census", "popcenstaircase.png", "Staircase"),
                    _img("population-census", "popcendiary1.png", "PopCenDiary1"),
                    _img("population-census", "populationcensuslastofficeleftdeskspawn.png", "Last Office Left Desk Spawn"),
                    _img("population-census", "spawn2.png", "Spawn2"),
                    _img("population-census", "archiveroomstaircase.png", "ArchiveRoomStaircase"),
                    _img("population-census", "populationcensussecondfloorunderfallenshelf.png", "Second Floor Under Fallen Shelf"),
                    _img("population-census", "popcens_5th_spawn.png", "5th Spawn"),
                ],
            },
        },
    },
    "64f731ab83cfca080a361e42": {
        "normalizedName": "pets-wont-need-it-part-1",
        "images": [],
        "objectives": {
            "64f731fa39e45b527a7c4300": {
                "type": "visit",
                "description": "Locate and scout the vet clinic on Streets of Tarkov",
                "images": [
                    _img("pets-wont-need-it-part-1", "the_outside_of_the_vet_clinic.png", "The outside of the vet clinic."),
                    _img("pets-wont-need-it-part-1", "the_front_door_of_the_vet_clinic.png", "The front door of the vet clinic."),
                    _img("pets-wont-need-it-part-1", "the_stairs_leading_to_the_2nd_floor.png", "The stairs leading to the 2nd floor."),
                    _img("pets-wont-need-it-part-1", "enter_the_pen_area_to_get_the_objective_completion.png", "Enter the pen area to get the objective completion."),
                ],
            },
            "64f732240e186112c4455d84": {
                "type": "visit",
                "description": "Locate and scout the x-ray technician's room on Streets of Tarkov",
                "images": [
                    _img("pets-wont-need-it-part-1", "the_outside_of_the_polyclinic.png", "The outside of the polyclinic."),
                    _img("pets-wont-need-it-part-1", "go_upstairs_and_open_the_left_door.png", "Go upstairs and open the left door."),
                    _img("pets-wont-need-it-part-1", "walk_down_the_corridor_and_go_through_the_second_door.png", "Walk down the corridor and go through the second door."),
                    _img("pets-wont-need-it-part-1", "you_ll_get_into_the_x-ray_room_where_you_can_unlock_the_x-ray_technician_s_room.png", "You'll get into the x-ray room where you can unlock the x-ray technician's room."),
                ],
            },
        },
    },
    "6573387d0b26ed4fde798de3": {
        "normalizedName": "pets-wont-need-it-part-2",
        "images": [],
        "objectives": {
            "6573387d0b26ed4fde798de4": {
                "type": "visit",
                "description": "Locate the first pharmacy on Primorsky Ave on Streets of Tarkov",
                "images": [
                    _img("pets-wont-need-it-part-2", "1._on_primorsky_ave_across_the_street_from_the_pinewood_hotel.png", "Across the street from the Pinewood hotel."),
                ],
            },
            "657338ded9b444e1fb07b1a8": {
                "type": "visit",
                "description": "Locate the second pharmacy on Primorsky Ave on Streets of Tarkov",
                "images": [
                    _img("pets-wont-need-it-part-2", "2._on_primorsky_ave_across_the_street_from_the_lexos_complex.png", "Across the street from the LexOs complex."),
                ],
            },
            "657338ed8e8e3a05445b760d": {
                "type": "visit",
                "description": "Locate the third pharmacy at Cardinal apartment complex on Streets of Tarkov",
                "images": [
                    _img("pets-wont-need-it-part-2", "3._in_the_24_7_supermarket_building_north_of_the_cardinal_apartment_complex.png", "3. In the 24/7 supermarket building, north of the cardinal apartment complex."),
                ],
            },
        },
    },
    "5ac3462b86f7741d6118b983": {
        "normalizedName": "farming-part-3",
        "images": [],
        "objectives": {
            "5ac6240786f77417204ca2b9": {
                "type": "visit",
                "description": "Locate the warehouse of seized goods on Customs",
                "images": [
                    _img("farming-part-3", "Customs-Factory-Outside.webp", "The warehouse"),
                ],
            },
        },
    },
    "6391359b9444fb141f4e6ee6": {
        "normalizedName": "youve-got-mail",
        "images": [],
        "objectives": {
            "6391359b9444fb141f4e6ee7": {
                "type": "findQuestItem",
                "description": "Obtain the registered mail on Streets of Tarkov",
                "images": [
                    _img("youve-got-mail", "the_post_office.png", "The post office"),
                    _img("youve-got-mail", "youvegotmailspawn01_01.png", "YouveGotMailSpawn01 01.png"),
                    _img("youve-got-mail", "youvegotmailspawn01_02.png", "YouveGotMailSpawn01 02.png"),
                    _img("youve-got-mail", "youvegotmailspawn01_03.png", "YouveGotMailSpawn01 03.png"),
                    _img("youve-got-mail", "youvegotmailspawn02_01.png", "YouveGotMailSpawn02 01.png"),
                    _img("youve-got-mail", "you_ve_got_mail_spawn1.png", "You've Got Mail Spawn1.png"),
                    _img("youve-got-mail", "youvegotmail_4thspawn.jpeg", "Youvegotmail 4thspawn.jpeg"),
                    _img("youve-got-mail", "you_ve_got_mail_spawn2.png", "You've Got Mail Spawn2.png"),
                    _img("youve-got-mail", "postofficestoragespawn.png", "PostOfficeStorageSpawn.png"),
                    _img("youve-got-mail", "postofficestoragespawn2.png", "PostOfficeStorageSpawn2.png"),
                ],
            },
        },
    },
    "63913715f8e5dd32bf4e3aaa": {
        "normalizedName": "broadcast-part-2",
        "images": [],
        "objectives": {
            "63a7d99804d3dc28a52a2106": {
                "type": "visit",
                "description": "Locate the place of bloody broadcast inside the grocery store on Streets of Tarkov",
                "images": [
                    _img("broadcast-part-2", "the_sparja_store.png", "The Sparja store"),
                    _img("broadcast-part-2", "broadcast_location_as_seen_from_main_entrance.png", "Broadcast location as seen from main entrance"),
                    _img("broadcast-part-2", "the_broadcast_location.png", "The broadcast location"),
                ],
            },
        },
    },
    "64ee9df4496db64f9b7a4432": {
        "normalizedName": "the-door",
        "images": [],
        "objectives": {
            "64ee9df4496db64f9b7a4433": {
                "type": "visit",
                "description": "Locate the cause of the signal jamming on Streets of Tarkov",
                "images": [
                    _img("the-door", "cosmonaut_painting_on_the_south_side_of_the_building.png", "Cosmonaut painting on the south side of the building"),
                    _img("the-door", "main_entrance_to_the_building.png", "Main entrance to the building"),
                    _img("the-door", "back_entrance_to_the_building.png", "Back entrance to the building"),
                    _img("the-door", "third_to_fourth_floor_stairs_with_trash_that_has_to_be_climbed_over.png", "Third to fourth floor stairs with trash that has to be climbed over"),
                    _img("the-door", "locked_door_on_the_fourth_floor.png", "Locked door on the fourth floor"),
                    _img("the-door", "chained_door_in_the_locked_area_where_the_first_camera_has_to_be_placed_at_the_heater.png", "Chained door in the locked area where the first camera has to be placed at the heater"),
                    _img("the-door", "second_placement_location_at_the_trash_bags_in_the_staircase_on_the_fourth_floor.png", "Second placement location at the trash bags in the staircase on the fourth floor"),
                ],
            },
            "64ee9df4496db64f9b7a4435": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera to watch the door",
                "images": [
                    _img("the-door", "cosmonaut_painting_on_the_south_side_of_the_building.png", "Cosmonaut painting on the south side of the building"),
                    _img("the-door", "main_entrance_to_the_building.png", "Main entrance to the building"),
                    _img("the-door", "back_entrance_to_the_building.png", "Back entrance to the building"),
                    _img("the-door", "third_to_fourth_floor_stairs_with_trash_that_has_to_be_climbed_over.png", "Third to fourth floor stairs with trash that has to be climbed over"),
                    _img("the-door", "locked_door_on_the_fourth_floor.png", "Locked door on the fourth floor"),
                    _img("the-door", "chained_door_in_the_locked_area_where_the_first_camera_has_to_be_placed_at_the_heater.png", "Chained door in the locked area where the first camera has to be placed at the heater"),
                    _img("the-door", "second_placement_location_at_the_trash_bags_in_the_staircase_on_the_fourth_floor.png", "Second placement location at the trash bags in the staircase on the fourth floor"),
                ],
            },
            "64eea6059878a0569d6ec98b": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera to watch the stairwell",
                "images": [
                    _img("the-door", "cosmonaut_painting_on_the_south_side_of_the_building.png", "Cosmonaut painting on the south side of the building"),
                    _img("the-door", "main_entrance_to_the_building.png", "Main entrance to the building"),
                    _img("the-door", "back_entrance_to_the_building.png", "Back entrance to the building"),
                    _img("the-door", "third_to_fourth_floor_stairs_with_trash_that_has_to_be_climbed_over.png", "Third to fourth floor stairs with trash that has to be climbed over"),
                    _img("the-door", "locked_door_on_the_fourth_floor.png", "Locked door on the fourth floor"),
                    _img("the-door", "chained_door_in_the_locked_area_where_the_first_camera_has_to_be_placed_at_the_heater.png", "Chained door in the locked area where the first camera has to be placed at the heater"),
                    _img("the-door", "second_placement_location_at_the_trash_bags_in_the_staircase_on_the_fourth_floor.png", "Second placement location at the trash bags in the staircase on the fourth floor"),
                ],
            },
        },
    },
    "67a0964e972c11a3f507731b": {
        "normalizedName": "needle-in-a-haystack",
        "images": [],
        "objectives": {
            "67a0bc81fcbc1c559d09b58f": {
                "type": "visit",
                "description": "Investigate the courtyard park by TerraGroup HQ on Ground Zero",
                "images": [
                    _img("needle-in-a-haystack", "the_courtyard_park.png", "The courtyard park"),
                ],
            },
            "67a0bc83182c4b0c71edd0ce": {
                "type": "visit",
                "description": "Investigate the kart track on Interchange",
                "images": [
                    _img("needle-in-a-haystack", "the_kart_track.png", "The kart track"),
                ],
            },
            "67a0bc8848d9d2cbd274ffd9": {
                "type": "visit",
                "description": "Investigate the coast-side construction site on Shoreline",
                "images": [
                    _img("needle-in-a-haystack", "the_construction_site.png", "The construction site"),
                ],
            },
        },
    },
    "639135e0fa894f0a866afde6": {
        "normalizedName": "urban-medicine",
        "images": [],
        "objectives": {
            "63a7d64710b7a13eb015961f": {
                "type": "visit",
                "description": "Locate the chemical laboratory on Streets of Tarkov",
                "images": [
                    _img("urban-medicine", "the_garage.png", "The garage"),
                    _img("urban-medicine", "urbanmedicinethirdbedspawn.png", "UrbanMedicineThirdBedSpawn"),
                    _img("urban-medicine", "urban_medicine_5.png", "Urban Medicine 5"),
                    _img("urban-medicine", "urbanmedicinefifthsixthshelfspawn.png", "UrbanMedicineFifthSixthShelfSpawn"),
                    _img("urban-medicine", "urban.png", "Urban"),
                    _img("urban-medicine", "urbanmedicine_spawn06.png", "UrbanMedicine Spawn06"),
                    _img("urban-medicine", "urban_medicine_spawn_2.png", "Urban Medicine Spawn 2"),
                    _img("urban-medicine", "urbanmedicinefourthshelfspawn.png", "UrbanMedicineFourthShelfSpawn"),
                    _img("urban-medicine", "urbanmedicinespawn05_02.png", "UrbanMedicineSpawn05 02"),
                    _img("urban-medicine", "urbanmedicinetrolleyspawn1.png", "UrbanMedicineTrolleySpawn1"),
                    _img("urban-medicine", "urbanmedicinetrolleyspawn2.png", "UrbanMedicineTrolleySpawn2"),
                    _img("urban-medicine", "urbanmedicinefirstbedspawn.png", "UrbanMedicineFirstBedSpawn"),
                    _img("urban-medicine", "urban_medicine_3.png", "Urban Medicine 3"),
                    _img("urban-medicine", "urbanmedicineseventhbedspawn.png", "UrbanMedicineSeventhBedSpawn"),
                    _img("urban-medicine", "urbanmedicine4thspawn2.png", "UrbanMedicine4thspawn2"),
                    _img("urban-medicine", "urbanmedicine10.png", "UrbanMedicine10"),
                    _img("urban-medicine", "urbanmedicinesecondboxspawn.png", "UrbanMedicineSecondBoxSpawn"),
                    _img("urban-medicine", "urban_medicine_4.png", "Urban Medicine 4"),
                ],
            },
            "639135e0fa894f0a866afde7": {
                "type": "findQuestItem",
                "description": "Obtain the container with drug samples",
                "images": [
                    _img("urban-medicine", "the_garage.png", "The garage"),
                    _img("urban-medicine", "urbanmedicinethirdbedspawn.png", "UrbanMedicineThirdBedSpawn"),
                    _img("urban-medicine", "urban_medicine_5.png", "Urban Medicine 5"),
                    _img("urban-medicine", "urbanmedicinefifthsixthshelfspawn.png", "UrbanMedicineFifthSixthShelfSpawn"),
                    _img("urban-medicine", "urban.png", "Urban"),
                    _img("urban-medicine", "urbanmedicine_spawn06.png", "UrbanMedicine Spawn06"),
                    _img("urban-medicine", "urban_medicine_spawn_2.png", "Urban Medicine Spawn 2"),
                    _img("urban-medicine", "urbanmedicinefourthshelfspawn.png", "UrbanMedicineFourthShelfSpawn"),
                    _img("urban-medicine", "urbanmedicinespawn05_02.png", "UrbanMedicineSpawn05 02"),
                    _img("urban-medicine", "urbanmedicinetrolleyspawn1.png", "UrbanMedicineTrolleySpawn1"),
                    _img("urban-medicine", "urbanmedicinetrolleyspawn2.png", "UrbanMedicineTrolleySpawn2"),
                    _img("urban-medicine", "urbanmedicinefirstbedspawn.png", "UrbanMedicineFirstBedSpawn"),
                    _img("urban-medicine", "urban_medicine_3.png", "Urban Medicine 3"),
                    _img("urban-medicine", "urbanmedicineseventhbedspawn.png", "UrbanMedicineSeventhBedSpawn"),
                    _img("urban-medicine", "urbanmedicine4thspawn2.png", "UrbanMedicine4thspawn2"),
                    _img("urban-medicine", "urbanmedicine10.png", "UrbanMedicine10"),
                    _img("urban-medicine", "urbanmedicinesecondboxspawn.png", "UrbanMedicineSecondBoxSpawn"),
                    _img("urban-medicine", "urban_medicine_4.png", "Urban Medicine 4"),
                ],
            },
        },
    },
    "639135f286e646067c176a87": {
        "normalizedName": "revision-streets-of-tarkov",
        "images": [],
        "objectives": {
            "63927f14744e452011470816": {
                "type": "mark",
                "description": "Locate and mark the first LAV III with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("revision-streets-of-tarkov", "lav_iii_1.png", "LAV III 1"),
                ],
            },
            "63927f2d8ba6894d155e77e6": {
                "type": "mark",
                "description": "Locate and mark the Stryker with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("revision-streets-of-tarkov", "stryker.png", "Stryker"),
                ],
            },
            "63927f349444fb141f4e6f24": {
                "type": "mark",
                "description": "Locate and mark the second LAV III with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("revision-streets-of-tarkov", "lav_iii_2.png", "LAV III 2"),
                ],
            },
        },
    },
    "638fcd23dc65553116701d33": {
        "normalizedName": "audit",
        "images": [],
        "objectives": {
            "638fd070202cd55bee01ca11": {
                "type": "findQuestItem",
                "description": "Obtain the financial records on Streets of Tarkov",
                "images": [
                    _img("audit", "the_building.png", "The building"),
                    _img("audit", "staircase_to_the_second_floor.png", "Staircase to the second floor"),
                    _img("audit", "the_wanted_office.png", "The wanted office"),
                    _img("audit", "the_wanted_file_folder.png", "The wanted file folder"),
                    _img("audit", "notes_inside_the_file_folder.png", "Notes inside the file folder"),
                ],
            },
        },
    },
    "639135b04ed9512be67647d7": {
        "normalizedName": "glory-to-cpsu-part-1",
        "images": [],
        "objectives": {
            "63a7db4a3e491955e65fb88b": {
                "type": "visit",
                "description": "Locate the apartment of Prapor's friend on Streets of Tarkov",
                "images": [
                    _img("glory-to-cpsu-part-1", "the_apartment_building_with_entrance_in_the_wall_marked.png", "The apartment building with entrance in the wall marked"),
                    _img("glory-to-cpsu-part-1", "way_to_the_western_staircase_1.png", "Way to the western staircase 1"),
                    _img("glory-to-cpsu-part-1", "way_to_the_western_staircase_2.png", "Way to the western staircase 2"),
                    _img("glory-to-cpsu-part-1", "way_to_the_western_staircase_3.png", "Way to the western staircase 3"),
                    _img("glory-to-cpsu-part-1", "the_western_staircase.png", "The western staircase"),
                    _img("glory-to-cpsu-part-1", "third_floor_hallway_with_hole_in_cell_5_floor_marked.png", "Third floor hallway with hole in cell #5 floor marked"),
                    _img("glory-to-cpsu-part-1", "second_floor_hallway_with_apartment_marked.png", "Second floor hallway with apartment marked"),
                    _img("glory-to-cpsu-part-1", "the_apartment.png", "The apartment"),
                ],
            },
        },
    },
    "64f3176921045e77405d63b5": {
        "normalizedName": "ambulances-again",
        "images": [],
        "objectives": {
            "64f3176921045e77405d63ba": {
                "type": "findQuestItem",
                "description": "Locate and obtain the ambulance paramedic's smartphone on Streets of Tarkov",
                "images": [
                    _img("ambulances-again", "ambulances_again_7.png"),
                    _img("ambulances-again", "ambulances_again_7_location.png"),
                    _img("ambulances-again", "ambulances_again_4.png"),
                    _img("ambulances-again", "ambulances_again_4_location.png"),
                    _img("ambulances-again", "ambulances_again_1.png"),
                    _img("ambulances-again", "ambulances_again_1.1.png"),
                    _img("ambulances-again", "ambulances_again_3.png"),
                    _img("ambulances-again", "ambulances_again_3_location.png"),
                    _img("ambulances-again", "ambulances_again_2.png"),
                    _img("ambulances-again", "ambulances_again_2.1.png"),
                    _img("ambulances-again", "ambulances_again_5.png"),
                    _img("ambulances-again", "ambulances_again_5.1.png"),
                    _img("ambulances-again", "ambulances_again_8.png"),
                    _img("ambulances-again", "ambulances_again_8.1.png"),
                    _img("ambulances-again", "ambulances_again_6.png"),
                    _img("ambulances-again", "ambulances_again_6_location.png"),
                ],
            },
        },
    },
    "639136d68ba6894d155e77cf": {
        "normalizedName": "green-corridor-bear",
        "images": [],
        "objectives": {
            "63a7d53904d3dc28a52a2100": {
                "type": "visit",
                "description": "Locate the cargo convoy on Streets of Tarkov",
                "images": [
                    _img("green-corridor-bear", "the_convoy.png", "The convoy"),
                ],
            },
            "63a7d58004d3dc28a52a2102": {
                "type": "visit",
                "description": "Determine the ambush spot",
                "images": [
                    _img("green-corridor-bear", "the_ambush_spot.png", "The ambush spot"),
                ],
            },
        },
    },
    "639282134ed9512be67647ed": {
        "normalizedName": "road-closed-usec",
        "images": [],
        "objectives": {
            "63a7d444f32fa1316250c3d5": {
                "type": "visit",
                "description": "Locate the cargo convoy on Streets of Tarkov",
                "images": [
                    _img("road-closed-usec", "the_convoy.png", "The convoy"),
                ],
            },
            "63a7d461f32fa1316250c3d7": {
                "type": "visit",
                "description": "Determine the ambush spot",
                "images": [
                    _img("road-closed-usec", "the_ambush_spot.png", "The ambush spot"),
                ],
            },
        },
    },
    "639136fa9444fb141f4e6eee": {
        "normalizedName": "watching-you",
        "images": [],
        "objectives": {
            "63a7d8665199ab1f7d4a7747": {
                "type": "visit",
                "description": "Locate the surveillance spot on Streets of Tarkov",
                "images": [
                    _img("watching-you", "northern_entrance.png", "Northern entrance"),
                    _img("watching-you", "door_to_the_room.png", "Door to the room"),
                ],
            },
            "639136fa9444fb141f4e6eef": {
                "type": "findQuestItem",
                "description": "Obtain the observation results",
                "images": [
                    _img("watching-you", "spawn_location_1_on_the_window_sill_next_to_the_book.png", "Spawn location 1 on the window sill next to the book"),
                    _img("watching-you", "spawn_location_2_on_the_chest_of_drawers.png", "Spawn location 2 on the chest of drawers"),
                ],
            },
        },
    },
    "63a511ea30d85e10e375b045": {
        "normalizedName": "broadcast-part-3",
        "images": [],
        "objectives": {
            "63ac22b8f627f540861d1b2d": {
                "type": "visit",
                "description": "Locate the place of bloody broadcast inside the old house on Streets of Tarkov",
                "images": [
                    _img("broadcast-part-3", "the_restaurant.png", "The restaurant"),
                    _img("broadcast-part-3", "the_broadcast_location.png", "The broadcast location"),
                ],
            },
        },
    },
    "639135a7e705511c8a4a1b78": {
        "normalizedName": "ballet-lover",
        "images": [],
        "objectives": {
            "63a7da6f5199ab1f7d4a774a": {
                "type": "visit",
                "description": "Locate the balletmeister's apartment on Streets of Tarkov",
                "images": [
                    _img("ballet-lover", "apartment_10_from_outside.png", "Needs Primorsky 46-48 skybridge key. Apartment 10 from outside"),
                    _img("ballet-lover", "the_skybridge_from_the_ground.png", "The skybridge from the ground"),
                    _img("ballet-lover", "way_to_the_primorsky_46_staircase_1.png", "Way to the Primorsky 46 staircase 1"),
                    _img("ballet-lover", "way_to_the_primorsky_46_staircase_2.png", "Way to the Primorsky 46 staircase 2"),
                    _img("ballet-lover", "way_to_the_primorsky_46_staircase_3.png", "Way to the Primorsky 46 staircase 3"),
                    _img("ballet-lover", "primorsky_46_staircase.png", "Primorsky 46 staircase"),
                    _img("ballet-lover", "door_to_the_skybridge_on_the_top_floor.png", "Door to the skybridge on the top floor"),
                    _img("ballet-lover", "the_skybridge_with_the_unlockable_door.png", "The skybridge with the unlockable door"),
                    _img("ballet-lover", "door_to_the_primorsky_48_apartment_10.png", "Door to the Primorsky 48 apartment 10"),
                    _img("ballet-lover", "inside_the_balletmeister_s_apartment.png", "Inside the balletmeister's apartment"),
                ],
            },
        },
    },
    "674600a366e6a521aa05eb66": {
        "normalizedName": "route-deviation",
        "images": [],
        "objectives": {
            "67460118d3498f1b35e0a025": {
                "type": "mark",
                "description": "Mark the Collapsed Crane BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "collapsed_crane_at_the_cabin_next_to_the_concrete_mixer.png", "Collapsed Crane: At the cabin next to the concrete mixer"),
                ],
            },
            "6746011dfd1dc9d0f502e55d": {
                "type": "mark",
                "description": "Mark the Old Scav Checkpoint BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "old_scav_checkpoint_at_the_corner_in_front_of_the_golden_world_store.png", "Old Scav Checkpoint: At the corner in front of the Golden World store"),
                ],
            },
            "674601247aa943781a1cf3fc": {
                "type": "mark",
                "description": "Mark the Pinewood Hotel BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "pinewood_hotel_at_the_trash_bags_next_to_one_of_the_pillars.png", "Pinewood Hotel: At the trash bags next to one of the pillars"),
                ],
            },
            "674601282043d1ef3c6b2eec": {
                "type": "mark",
                "description": "Mark the City Center BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "city_center_at_the_traffic_light_on_the_corner_of_the_primorsky_49_building.png", "City Center: At the traffic light on the corner of the Primorsky 49 building"),
                ],
            },
            "6746012a35218bb89951248e": {
                "type": "mark",
                "description": "Mark the Tram BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "tram_at_the_barrier_in_front_of_the_restaurant.png", "Tram: At the barrier in front of the restaurant"),
                ],
            },
            "6746012d871e69a9abb5873d": {
                "type": "mark",
                "description": "Mark the Rodina Cinema BTR stop with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("route-deviation", "rodina_cinema_at_the_container_under_the_stairs.png", "Rodina Cinema: At the container under the stairs"),
                ],
            },
        },
    },
    "64f5aac4b63b74469b6c14c2": {
        "normalizedName": "glory-to-cpsu-part-2",
        "images": [],
        "objectives": {
            "64f5aac4b63b74469b6c14c7": {
                "type": "visit",
                "description": "Locate the place of work of Prapor's friend on Streets of Tarkov",
                "images": [
                    _img("glory-to-cpsu-part-2", "the_museum_building.png", "The museum building"),
                    _img("glory-to-cpsu-part-2", "way_to_the_stairs.png", "Way to the stairs"),
                    _img("glory-to-cpsu-part-2", "way_to_the_room_with_the_spawns.png", "Way to the room with the spawns"),
                    _img("glory-to-cpsu-part-2", "journalspawn1.png", "JournalSpawn1.png"),
                    _img("glory-to-cpsu-part-2", "cpsu_part_2.png", "CPSU Part 2.png"),
                    _img("glory-to-cpsu-part-2", "cpsu2_3rd_spawn.png", "CPSU2 3rd spawn.png"),
                    _img("glory-to-cpsu-part-2", "journalspawn2.png", "JournalSpawn2.png"),
                ],
            },
            "64f5b876a2a5bb727b61c5ad": {
                "type": "findQuestItem",
                "description": "Obtain any information on the fate of Prapor's friend",
                "images": [
                    _img("glory-to-cpsu-part-2", "the_museum_building.png", "The museum building"),
                    _img("glory-to-cpsu-part-2", "way_to_the_stairs.png", "Way to the stairs"),
                    _img("glory-to-cpsu-part-2", "way_to_the_room_with_the_spawns.png", "Way to the room with the spawns"),
                    _img("glory-to-cpsu-part-2", "journalspawn1.png", "JournalSpawn1.png"),
                    _img("glory-to-cpsu-part-2", "cpsu_part_2.png", "CPSU Part 2.png"),
                    _img("glory-to-cpsu-part-2", "cpsu2_3rd_spawn.png", "CPSU2 3rd spawn.png"),
                    _img("glory-to-cpsu-part-2", "journalspawn2.png", "JournalSpawn2.png"),
                ],
            },
        },
    },
    "64f5e20652fc01298e2c61e3": {
        "normalizedName": "beyond-the-red-meat-part-1",
        "images": [],
        "objectives": {
            "64f6a9e6dd44b6417729b535": {
                "type": "findQuestItem",
                "description": "Locate and obtain information on culinary secrets at the Beluga restaurant on Streets of Tarkov",
                "images": [
                    _img("beyond-the-red-meat-part-1", "beyondtheredmeatspawn0201.png", "BeyondTheRedMeatSpawn0201.png"),
                    _img("beyond-the-red-meat-part-1", "beyondtheredmeatspawn0202.png", "BeyondTheRedMeatSpawn0202.png"),
                    _img("beyond-the-red-meat-part-1", "beyond_the_red_meat_part_1_3rd_spawn_far.png", "Beyond the Red Meat Part 1 3rd spawn far.png"),
                    _img("beyond-the-red-meat-part-1", "beyond_the_red_meat_part_1_3rd_spawn.png", "Beyond the Red Meat Part 1 3rd spawn.png"),
                    _img("beyond-the-red-meat-part-1", "beyondtheredmeatspawn0101.png", "BeyondTheRedMeatSpawn0101.png"),
                    _img("beyond-the-red-meat-part-1", "beyondtheredmeatspawn0102.png", "BeyondTheRedMeatSpawn0102.png"),
                    _img("beyond-the-red-meat-part-1", "the_red_meat_part_1_4th_spawn_far.png", "The Red Meat Part 1 4th spawn far.png"),
                    _img("beyond-the-red-meat-part-1", "the_red_meat_part_1_4th_spawn_close.png", "The Red Meat Part 1 4th spawn close.png"),
                ],
            },
        },
    },
    "64f6aafd67e11a7c6206e0d0": {
        "normalizedName": "beyond-the-red-meat-part-2",
        "images": [],
        "objectives": {
            "64f6aafd67e11a7c6206e0d1": {
                "type": "findQuestItem",
                "description": "Locate and obtain the secret ingredient at the TerraGroup office on Streets of Tarkov",
                "images": [
                    _img("beyond-the-red-meat-part-2", "redmeat2buildingentrance.png", "RedMeat2BuildingEntrance"),
                    _img("beyond-the-red-meat-part-2", "terragroup_security_armory_way_1.png", "TerraGroup security armory Way 1"),
                    _img("beyond-the-red-meat-part-2", "terragroup_security_armory_way_2.png", "TerraGroup security armory Way 2"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_location_1.png", "Beyond the Red Meat Location 1"),
                    _img("beyond-the-red-meat-part-2", "terragroup_meetingroom_lockeddoor.png", "TerraGroup MeetingRoom LockedDoor"),
                    _img("beyond-the-red-meat-part-2", "2023-09-07-14-30-_44.3_1.7_-59.4_0.1_-0.2_0.0_1.0_6.51_0.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_location_2_2.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_2_31.png"),
                    _img("beyond-the-red-meat-part-2", "image_51.png"),
                    _img("beyond-the-red-meat-part-2", "image_52.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_2_42.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_2_43.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_2_32.png"),
                    _img("beyond-the-red-meat-part-2", "beyond_the_red_meat_2_33.png"),
                ],
            },
        },
    },
    "6573397ef3f8344c4575cd87": {
        "normalizedName": "properties-all-around",
        "images": [],
        "objectives": {
            "6573397ef3f8344c4575cd88": {
                "type": "visit",
                "description": "Locate the real estate fund on Streets of Tarkov",
                "images": [
                    _img("properties-all-around", "the_real_estate_agency_building.png", "Not all spawn locations for the quest item requires REA key. The real estate agency building"),
                    _img("properties-all-around", "properties_all_around_spawn_1.png", "Properties All Around Spawn 1.png"),
                    _img("properties-all-around", "properties_all_around_spawn_2.png", "Properties All Around Spawn 2.png"),
                    _img("properties-all-around", "properties_all_around_spawn_3.png", "Properties All Around Spawn 3.png"),
                    _img("properties-all-around", "properties_all_around_spawn_4.png", "Properties All Around Spawn 4.png"),
                    _img("properties-all-around", "properties_all_around_spawn_5.png", "Properties All Around Spawn 5.png"),
                    _img("properties-all-around", "properties_all_around_spawn_6.png", "Properties All Around Spawn 6.png"),
                ],
            },
            "6581676e7a18ff402fd23e68": {
                "type": "findQuestItem",
                "description": "Locate and obtain the Tarkov real estate transactions document",
                "images": [
                    _img("properties-all-around", "the_real_estate_agency_building.png", "Not all spawn locations for the quest item requires REA key. The real estate agency building"),
                    _img("properties-all-around", "properties_all_around_spawn_1.png", "Properties All Around Spawn 1.png"),
                    _img("properties-all-around", "properties_all_around_spawn_2.png", "Properties All Around Spawn 2.png"),
                    _img("properties-all-around", "properties_all_around_spawn_3.png", "Properties All Around Spawn 3.png"),
                    _img("properties-all-around", "properties_all_around_spawn_4.png", "Properties All Around Spawn 4.png"),
                    _img("properties-all-around", "properties_all_around_spawn_5.png", "Properties All Around Spawn 5.png"),
                    _img("properties-all-around", "properties_all_around_spawn_6.png", "Properties All Around Spawn 6.png"),
                ],
            },
        },
    },
    "66d9cbb67b491f9d5304f6e6": {
        "normalizedName": "is-this-a-reference",
        "images": [],
        "objectives": {
            "66d9cbb67b491f9d5304f6e9": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the amphibian pizza lovers' hideout on Streets of Tarkov",
                "images": [
                    _img("is-this-a-reference", "itarpizzalover.png", "At Sewer River extraction"),
                ],
            },
            "66d9cbb67b491f9d5304f6ea": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the burned girl's sickroom on Streets of Tarkov",
                "images": [
                    _img("is-this-a-reference", "x-ray_clinic_outside.png", "X-ray clinic outside"),
                    _img("is-this-a-reference", "x-ray_key_upstairs.png", "X-ray key upstairs"),
                    _img("is-this-a-reference", "itarburnedgirldoor.png", "On the second floor"),
                    _img("is-this-a-reference", "itarburnedgirl.png", "On the second floor"),
                ],
            },
            "66d9cbb67b491f9d5304f6eb": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the barbed wire body on Streets of Tarkov",
                "images": [
                    _img("is-this-a-reference", "mysterious_key_building_entrance.png", "Mysterious key building entrance"),
                    _img("is-this-a-reference", "mysterious_key_path1.png", "path1"),
                    _img("is-this-a-reference", "mysterious_key_path2.png", "path2"),
                    _img("is-this-a-reference", "mysterious_key_path3.png", "path3"),
                    _img("is-this-a-reference", "itarwirebodydoor.png", "Door"),
                    _img("is-this-a-reference", "itarwirebody.png", "Barbed Wire Body"),
                ],
            },
            "66dacf2a88c7001436a67390": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the really scary hole in the wall on Streets of Tarkov",
                "images": [
                    _img("is-this-a-reference", "mysterious_key_building_entrance.png", "Mysterious key building entrance.png"),
                    _img("is-this-a-reference", "itarscaryhole.png", "The Scary Hole"),
                ],
            },
            "66dad1d7811532f53e472c13": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the high rise where one die-hard guy kicked ass on Ground Zero",
                "images": [
                    _img("is-this-a-reference", "itardiehardguy.png"),
                ],
            },
            "66dad1d941756590432b0eaa": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the computer with a joke for programmers on Ground Zero",
                "images": [
                    _img("is-this-a-reference", "itarjokeforprogrammers.png"),
                ],
            },
            "66dad1dbb16caebe0e214d89": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the little chair surrounded by big black chairs on Ground Zero",
                "images": [
                    _img("is-this-a-reference", "tarbank_building.png", "Tarbank Building"),
                    _img("is-this-a-reference", "tarbank_building_stairs.png", "Go Upstairs jump barb wire"),
                    _img("is-this-a-reference", "itarchairs.png", "WC Surrounded by BBCs"),
                ],
            },
            "66dad1ddec53b0df3b10a1b9": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera by your friend Wilson on Lighthouse",
                "images": [
                    _img("is-this-a-reference", "itarfriendwilson.png", "Near the western houses"),
                ],
            },
            "66dad1de93c8fcffd5790d89": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the kindled rest place on Lighthouse",
                "images": [
                    _img("is-this-a-reference", "itarkindledplace.png", "The kindled rest place"),
                ],
            },
            "66dad1e0703b718902451ee4": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the ominous welcome sign on Lighthouse",
                "images": [
                    _img("is-this-a-reference", "itarwelcomesign.png", "Along the main road"),
                ],
            },
            "66dad1e21e7ef28d17a69a93": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the megagenius scientist's seat on Lighthouse",
                "images": [
                    _img("is-this-a-reference", "itarmegagenius.png", "Megagenius Scientist's Seat"),
                ],
            },
            "66dad1e607181e2f78a3a0a2": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the place where conscripts did a very important task on Reserve",
                "images": [
                    _img("is-this-a-reference", "itarconscriptstask.png", "At the Barracks"),
                ],
            },
            "66dad1e843a718561db0fdd3": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the two chair riddle on Reserve",
                "images": [
                    _img("is-this-a-reference", "itarchairriddle.png"),
                ],
            },
            "66dad1ebc5c8e6cd26dd1d31": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at every tank driver's first videogame on Reserve",
                "images": [
                    _img("is-this-a-reference", "itartankvideogame.png", "Southside wall"),
                ],
            },
            "66dad1edbc4fdd0c6eb38c5e": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the bucket-head villain on Customs",
                "images": [
                    _img("is-this-a-reference", "itarbuckethead.png", "Bathroom upstairs"),
                ],
            },
            "66dad1f00e049ac7abb6d801": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at Anvil 3-4 crew member's execution site on Customs",
                "images": [
                    _img("is-this-a-reference", "itaranvil34.png"),
                ],
            },
            "66dad1f22edc2103eb176de8": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at every indie developer's worst nightmare on Customs",
                "images": [
                    _img("is-this-a-reference", "itardevelopersnightmare.png"),
                ],
            },
            "66db1f8d7539f4b4eb640aff": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera in the room where the firefighter burned the books to become happy on Factory",
                "images": [
                    _img("is-this-a-reference", "itarfirefighterbooks.png"),
                ],
            },
            "66db1f8f564045697071d934": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the place where some valve technicians couldn't finish their third project on Factory",
                "images": [
                    _img("is-this-a-reference", "itarvalvetechnicians.png"),
                ],
            },
            "66db1f928d59a9fe511dfc25": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the dangerous Russian chocolate beauty on Factory",
                "images": [
                    _img("is-this-a-reference", "itarrussianbeauty.png", "In the locker room"),
                ],
            },
            "66db1f94a147d9840ec0dfba": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera in the conspiracy theorist's room on Factory",
                "images": [
                    _img("is-this-a-reference", "factorylockedroom.png", "Factory emergency exit key required"),
                    _img("is-this-a-reference", "itarconspiracytheorist.png"),
                ],
            },
            "66db1f98b8e22a92437fe5c6": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera near where someone thought he'd get transported to the magic school on Factory",
                "images": [
                    _img("is-this-a-reference", "itarmagicschool.png"),
                ],
            },
            "66db1f9b4f7bf01d937eb150": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the mother of all strategy games on Factory",
                "images": [
                    _img("is-this-a-reference", "itarstrategygames.png", "ITARStrategyGames.png"),
                ],
            },
            "66db1ff798d52a5a8e4ed1f8": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the stairway to the skies on Factory",
                "images": [
                    _img("is-this-a-reference", "itarstairwayskies.png", "Inside the Gate 3 extraction"),
                ],
            },
            "66debf2b9e4ce2ef233ee5b7": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the bear who sat into a flaming car on Woods",
                "images": [
                    _img("is-this-a-reference", "itarbearcar.png"),
                ],
            },
            "66debf2e1e254957b82711ff": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the upside-down chair on Shoreline",
                "images": [
                    _img("is-this-a-reference", "itarupsidedownchairhouse.png", "The door to the room has to be breached"),
                    _img("is-this-a-reference", "itarupsidedownchair.png", "The door to the room has to be breached"),
                ],
            },
            "66debf30802386a45d0adb60": {
                "type": "plantItem",
                "description": "Install a WI-FI Camera at the not-so-lonely bathroom on Shoreline",
                "images": [
                    _img("is-this-a-reference", "pierofficebuilding.png", "Pier Office Building"),
                    _img("is-this-a-reference", "itarnotsolonelybathroom.png", "not-so-lonely bathroom"),
                ],
            },
        },
    },
    "65733403eefc2c312a759ddb": {
        "normalizedName": "developers-secrets-part-1",
        "images": [],
        "objectives": {
            "65733403eefc2c312a759ddc": {
                "type": "visit",
                "description": "Locate the IT firm office on Streets of Tarkov",
                "images": [
                    _img("developers-secrets-part-1", "the_building.png", "The building"),
                    _img("developers-secrets-part-1", "the_negotiation_room.png", "A Negotiation room key is required to access it."),
                ],
            },
            "65801ad655315fdce2096bec": {
                "type": "visit",
                "description": "Unravel the secret of the firm's success",
                "images": [
                    _img("developers-secrets-part-1", "the_building.png", "The building"),
                    _img("developers-secrets-part-1", "the_negotiation_room.png", "A Negotiation room key is required to access it."),
                ],
            },
        },
    },
    "6573382e557ff128bf3da536": {
        "normalizedName": "developers-secrets-part-2",
        "images": [],
        "objectives": {
            "6573382e557ff128bf3da537": {
                "type": "visit",
                "description": "Unravel the secret of the IT firm's employees' high morale on Streets of Tarkov",
                "images": [
                    _img("developers-secrets-part-2", "door_to_the_hookah_lounge.png", "Need Relaxation room key - Door to the hookah lounge"),
                ],
            },
        },
    },
    "639135e8c115f907b14700aa": {
        "normalizedName": "surveillance",
        "images": [],
        "objectives": {
            "639135e8c115f907b14700ab": {
                "type": "findQuestItem",
                "description": "Obtain the data on vehicle movement at the Concordia residential complex parking lot on Streets of Tarkov",
                "images": [
                    _img("surveillance", "the_concordia_building.png", "The Concordia building"),
                    _img("surveillance", "way_to_the_parking_garage_via_house_no._1_staircase.png", "Way to the parking garage via house no. 1 staircase"),
                    _img("surveillance", "the_staircase.png", "The staircase"),
                    _img("surveillance", "door_location_near_the_staircase.png", "Door location near the staircase"),
                    _img("surveillance", "door_to_the_room.png", "Door to the room"),
                    _img("surveillance", "hddonpc.png", "On the computer under the left side of the corner desk"),
                    _img("surveillance", "hddunderdesk2.png", "Between the desktop and the drawers on the right side of the corner desk"),
                    _img("surveillance", "hddpizzabox.png", "In the pizza box on the safe in the back left"),
                    _img("surveillance", "hddunderdesk.png", "In the compartment under the right side desk"),
                ],
            },
        },
    },
    "639135bbc115f907b14700a6": {
        "normalizedName": "audiophile",
        "images": [],
        "objectives": {
            "63a7d8f404d3dc28a52a2104": {
                "type": "visit",
                "description": "Locate the musician gathering spot on Streets of Tarkov",
                "images": [
                    _img("audiophile", "apartment_10_from_outside.png", "Apartment 10 from outside. Bring flashlight. Need Primorsky 46-48 skybridge key."),
                    _img("audiophile", "the_skybridge_from_the_ground.png", "The skybridge from the ground"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_1.png", "Way to the Primorsky 46 staircase 1"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_2.png", "Way to the Primorsky 46 staircase 2"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_3.png", "Way to the Primorsky 46 staircase 3"),
                    _img("audiophile", "primorsky_46_staircase.png", "Primorsky 46 staircase"),
                    _img("audiophile", "door_to_the_skybridge_on_the_top_floor.png", "Door to the skybridge on the top floor"),
                    _img("audiophile", "the_skybridge_with_the_unlockable_door.png", "The skybridge with the unlockable door"),
                    _img("audiophile", "door_to_the_primorsky_48_apartment_10.png", "Door to the Primorsky 48 apartment 10"),
                    _img("audiophile", "door_to_the_music_room.png", "Door to the music room"),
                    _img("audiophile", "the_music_room.png", "The music room"),
                    _img("audiophile", "eg_pick_streets_spawn6.png", "Eg pick streets spawn6.png"),
                    _img("audiophile", "eg_pick_streets_spawn5.png", "Eg pick streets spawn5.png"),
                    _img("audiophile", "eg_pick_streets_spawn1.png", "Eg pick streets spawn1.png"),
                    _img("audiophile", "eg_pick_streets_spawn2.png", "Eg pick streets spawn2.png"),
                    _img("audiophile", "eg_pick_streets_spawn3.png", "EG pick streets spawn3.png"),
                    _img("audiophile", "eg_pick_streets_spawn8.png", "Eg pick streets spawn8.png"),
                    _img("audiophile", "audiophilespawn10.png", "AudiophileSpawn10.png"),
                    _img("audiophile", "eg_pick_streets_spawn9.png", "Eg pick streets spawn9.png"),
                    _img("audiophile", "eg_pick_streets_spawn7.png", "Eg pick streets spawn7.png"),
                    _img("audiophile", "eg_pick_streets_spawn4.png", "Eg pick streets spawn4.png"),
                ],
            },
            "639135bbc115f907b14700a7": {
                "type": "findQuestItem",
                "description": "Obtain the engraved guitar pick",
                "images": [
                    _img("audiophile", "apartment_10_from_outside.png", "Apartment 10 from outside. Bring flashlight. Need Primorsky 46-48 skybridge key."),
                    _img("audiophile", "the_skybridge_from_the_ground.png", "The skybridge from the ground"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_1.png", "Way to the Primorsky 46 staircase 1"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_2.png", "Way to the Primorsky 46 staircase 2"),
                    _img("audiophile", "way_to_the_primorsky_46_staircase_3.png", "Way to the Primorsky 46 staircase 3"),
                    _img("audiophile", "primorsky_46_staircase.png", "Primorsky 46 staircase"),
                    _img("audiophile", "door_to_the_skybridge_on_the_top_floor.png", "Door to the skybridge on the top floor"),
                    _img("audiophile", "the_skybridge_with_the_unlockable_door.png", "The skybridge with the unlockable door"),
                    _img("audiophile", "door_to_the_primorsky_48_apartment_10.png", "Door to the Primorsky 48 apartment 10"),
                    _img("audiophile", "door_to_the_music_room.png", "Door to the music room"),
                    _img("audiophile", "the_music_room.png", "The music room"),
                    _img("audiophile", "eg_pick_streets_spawn6.png", "Eg pick streets spawn6.png"),
                    _img("audiophile", "eg_pick_streets_spawn5.png", "Eg pick streets spawn5.png"),
                    _img("audiophile", "eg_pick_streets_spawn1.png", "Eg pick streets spawn1.png"),
                    _img("audiophile", "eg_pick_streets_spawn2.png", "Eg pick streets spawn2.png"),
                    _img("audiophile", "eg_pick_streets_spawn3.png", "EG pick streets spawn3.png"),
                    _img("audiophile", "eg_pick_streets_spawn8.png", "Eg pick streets spawn8.png"),
                    _img("audiophile", "audiophilespawn10.png", "AudiophileSpawn10.png"),
                    _img("audiophile", "eg_pick_streets_spawn9.png", "Eg pick streets spawn9.png"),
                    _img("audiophile", "eg_pick_streets_spawn7.png", "Eg pick streets spawn7.png"),
                    _img("audiophile", "eg_pick_streets_spawn4.png", "Eg pick streets spawn4.png"),
                ],
            },
        },
    },
    "6391372c8ba6894d155e77d7": {
        "normalizedName": "broadcast-part-4",
        "images": [],
        "objectives": {
            "63a7d6d61f06d111271f5aeb": {
                "type": "visit",
                "description": "Locate the cultist meeting spot on Streets of Tarkov",
                "images": [
                    _img("broadcast-part-4", "marked_room_in_the_building_and_basement_entrance_marked.png", "Marked room in the building and basement entrance marked"),
                    _img("broadcast-part-4", "view_to_the_marked_room_from_second_floor_staircase.png", "View to the marked room from second floor staircase"),
                    _img("broadcast-part-4", "door_to_the_marked_room.png", "Door to the marked room"),
                    _img("broadcast-part-4", "inside_the_marked_room.png", "Inside the marked room"),
                ],
            },
        },
    },
    "64ee99639878a0569d6ec8c9": {
        "normalizedName": "broadcast-part-5",
        "images": [],
        "objectives": {
            "64ee9a979878a0569d6ec8cb": {
                "type": "visit",
                "description": "Locate the ritual spot on Chekannaya st. on Streets of Tarkov",
                "images": [
                    _img("broadcast-part-5", "entrance_to_the_building_through_a_window_from_the_courtyard.png", "Entrance to the building through a window from the courtyard"),
                    _img("broadcast-part-5", "once_inside_get_in_the_corridor_and_move_to_the_second_room_on_the_left.png", "Once inside, get in the corridor and move to the second room on the left"),
                    _img("broadcast-part-5", "in_the_bedroom_go_through_the_hole_in_the_wall.png", "In the bedroom, go through the hole in the wall"),
                    _img("broadcast-part-5", "after_going_through_the_hole_get_downstairs_and_open_the_door_to_the_left.png", "After going through the hole, get downstairs and open the door to the left"),
                    _img("broadcast-part-5", "you_get_in_front_of_the_door_to_unlock.png", "You get in front of the door to unlock"),
                ],
            },
            "64ee9b5a1a5f313cb144bf87": {
                "type": "plantItem",
                "description": "Stash a Cultist knife at the ritual spot",
                "images": [
                    _img("broadcast-part-5", "entrance_to_the_building_through_a_window_from_the_courtyard.png", "Entrance to the building through a window from the courtyard"),
                    _img("broadcast-part-5", "once_inside_get_in_the_corridor_and_move_to_the_second_room_on_the_left.png", "Once inside, get in the corridor and move to the second room on the left"),
                    _img("broadcast-part-5", "in_the_bedroom_go_through_the_hole_in_the_wall.png", "In the bedroom, go through the hole in the wall"),
                    _img("broadcast-part-5", "after_going_through_the_hole_get_downstairs_and_open_the_door_to_the_left.png", "After going through the hole, get downstairs and open the door to the left"),
                    _img("broadcast-part-5", "you_get_in_front_of_the_door_to_unlock.png", "You get in front of the door to unlock"),
                ],
            },
        },
    },
    "666314c5a9290f9e0806cca5": {
        "normalizedName": "key-to-the-city",
        "images": [],
        "objectives": {
            "6679884cef969161e3e9d64d": {
                "type": "findQuestItem",
                "description": "Locate and obtain the souvenir key to the city on Streets of Tarkov",
                "images": [
                    _img("key-to-the-city", "the_museum_building.png", "The museum building"),
                    _img("key-to-the-city", "first_floor_key_locations_marked.png", "First floor key locations marked"),
                    _img("key-to-the-city", "1_in_a_display_case_against_the_inner_front_edge.png", "1: In a display case, against the inner front edge"),
                    _img("key-to-the-city", "2_in_a_display_case_behind_an_angled_display_stand.png", "2: In a display case, behind an angled display stand"),
                    _img("key-to-the-city", "3_in_a_display_case_in_the_inner_left_corner.png", "3: In a display case, in the inner left corner"),
                    _img("key-to-the-city", "4_in_a_display_case_under_a_book.png", "4: In a display case, under a book"),
                    _img("key-to-the-city", "way_to_the_stairs.png", "Way to the stairs"),
                    _img("key-to-the-city", "on_the_lowboard_in_the_hallway.png", "On the lowboard in the hallway"),
                    _img("key-to-the-city", "on_the_desk_in_the_room_with_the_kitchen.png", "On the desk in the room with the kitchen"),
                    _img("key-to-the-city", "in_the_middle_compartment_of_the_bedroom_bookshelf.png", "In the middle compartment of the bedroom bookshelf"),
                ],
            },
        },
    },
    "6572e876dc0d635f633a5714": {
        "normalizedName": "pyramid-scheme",
        "images": [],
        "objectives": {
            "6573057d66de94c756a60472": {
                "type": "visit",
                "description": "Locate the first ATM on Nizhnaya Sadovaya Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._14_-_first_atm_on_nizhnaya_sadovaya_street.png", "First ATM on Nizhnaya Sadovaya Street"),
                ],
            },
            "65730580123b33fa7bb1a9fc": {
                "type": "visit",
                "description": "Locate the second ATM on Nizhnaya Sadovaya Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._15_-_second_atm_on_nizhnaya_sadovaya_street.png", "Second ATM on Nizhnaya Sadovaya Street"),
                ],
            },
            "657305885511555648dbb952": {
                "type": "visit",
                "description": "Locate the ATM on Razvedchikov Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._13_-_atm_on_razvedchikov_street.png", "ATM on Razvedchikov Street"),
                ],
            },
            "65730579d50436e87723dae4": {
                "type": "visit",
                "description": "Locate the first ATM on Klimov Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._3_-_first_atm_on_klimov_street.png", "First ATM on Klimov Street"),
                ],
            },
            "6573058d6a542c099ce5d92b": {
                "type": "visit",
                "description": "Locate the second ATM on Klimov Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._11_-_second_atm_on_klimov_street.png", "Second ATM on Klimov Street"),
                ],
            },
            "657305913186869491d7f5fd": {
                "type": "visit",
                "description": "Locate the third ATM on Klimov Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._8_-_third_atm_on_klimov_street.png", "Third ATM on Klimov Street"),
                ],
            },
            "65730593d4b1bd9359203a74": {
                "type": "visit",
                "description": "Locate the fourth ATM on Klimov Street on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._9_-_fourth_atm_on_klimov_street.png", "Fourth ATM on Klimov Street"),
                ],
            },
            "6573058510b6071846639907": {
                "type": "visit",
                "description": "Locate the first ATM on Primorsky Ave on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._4_-_second_atm_on_primorsky_ave.png", "Second ATM on Primorsky Ave"),
                ],
            },
            "65730597d64537696cc661c6": {
                "type": "visit",
                "description": "Locate the second ATM on Primorsky Ave on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._4_-_second_atm_on_primorsky_ave.png", "Second ATM on Primorsky Ave"),
                ],
            },
            "657305989d1d1fe81237e9e7": {
                "type": "visit",
                "description": "Locate the third ATM on Primorsky Ave on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._5_-_third_atm_on_primorsky_ave.png", "Third ATM on Primorsky Ave"),
                ],
            },
            "6573059ace022903ee1e6b11": {
                "type": "visit",
                "description": "Locate the ATM at the expo on Streets of Tarkov",
                "images": [
                    _img("pyramid-scheme", "no._1_-_atm_at_the_expo.png", "ATM at the expo"),
                ],
            },
        },
    },
    "639dbaf17c898a131e1cffff": {
        "normalizedName": "debtor",
        "images": [],
        "objectives": {
            "63a7cd3aee7b4d0d5507baee": {
                "type": "visit",
                "description": "Find the debtor on Streets of Tarkov",
                "images": [
                    _img("debtor", "northern_entrance.png", "Northern entrance"),
                    _img("debtor", "door_to_the_room.png", "Door to the room"),
                    _img("debtor", "the_debtor_on_the_ground.png", "The debtor on the ground"),
                ],
            },
        },
    },
    "65734c186dc1e402c80dc19e": {
        "normalizedName": "dandies",
        "images": [],
        "objectives": {
            "657356c410becd24bc776f55": {
                "type": "plantItem",
                "description": "Stash a Bomber beanie inside the barber shop on Streets of Tarkov",
                "images": [
                    _img("dandies", "the_barber_shop.png", "The barber shop"),
                    _img("dandies", "first_stashing_location_at_the_table.png", "First stashing location at the table"),
                    _img("dandies", "second_stashing_location_at_the_left_side_chairs.png", "Second stashing location at the left side chairs"),
                ],
            },
            "657356d0a95a1e7e1a8d8d99": {
                "type": "plantItem",
                "description": "Stash RayBench Hipster Reserve sunglasses inside the barber shop on Streets of Tarkov",
                "images": [
                    _img("dandies", "the_barber_shop.png", "The barber shop"),
                    _img("dandies", "first_stashing_location_at_the_table.png", "First stashing location at the table"),
                    _img("dandies", "second_stashing_location_at_the_left_side_chairs.png", "Second stashing location at the left side chairs"),
                ],
            },
        },
    },
    "639135c3744e452011470807": {
        "normalizedName": "house-arrest-part-1",
        "images": [],
        "objectives": {
            "63a7d767f32fa1316250c3da": {
                "type": "visit",
                "description": "Locate where the missing group was held captive on Streets of Tarkov",
                "images": [
                    _img("house-arrest-part-1", "the_apartment_building_with_the_entrance_in_the_wall_marked.png", "The apartment building with the entrance in the wall marked"),
                    _img("house-arrest-part-1", "way_to_the_western_staircase_marked_on_building_map.png", "Way to the western staircase marked on building map"),
                    _img("house-arrest-part-1", "way_to_the_western_staircase_1.png", "Way to the western staircase 1"),
                    _img("house-arrest-part-1", "way_to_the_western_staircase_2.png", "Way to the western staircase 2"),
                    _img("house-arrest-part-1", "way_to_the_western_staircase_3.png", "Way to the western staircase 3"),
                    _img("house-arrest-part-1", "the_western_staircase.png", "The western staircase"),
                    _img("house-arrest-part-1", "third_floor_hallway_with_cell_no._1_marked.png", "Third floor hallway with cell no. 1 marked"),
                    _img("house-arrest-part-1", "the_prison_cell.png", "The prison cell"),
                ],
            },
        },
    },
    "639135cd8ba6894d155e77cb": {
        "normalizedName": "house-arrest-part-2",
        "images": [],
        "objectives": {
            "63a7dbc31943b749b5021ec3": {
                "type": "visit",
                "description": "Locate the improvised jail warden's apartment on Streets of Tarkov",
                "images": [
                    _img("house-arrest-part-2", "the_apartment_building_with_entrance_in_the_wall_marked.png", "The apartment building with entrance in the wall marked"),
                    _img("house-arrest-part-2", "way_to_the_eastern_staircase.png", "Way to the eastern staircase"),
                    _img("house-arrest-part-2", "the_warden_s_body.png", "The warden's body"),
                    _img("house-arrest-part-2", "way_to_the_western_staircase_1.png", "Way to the western staircase 1"),
                    _img("house-arrest-part-2", "way_to_the_western_staircase_2.png", "Way to the western staircase 2"),
                    _img("house-arrest-part-2", "way_to_the_western_staircase_3.png", "Way to the western staircase 3"),
                    _img("house-arrest-part-2", "the_western_staircase.png", "The western staircase"),
                    _img("house-arrest-part-2", "third_floor_hallway_with_hole_in_cell_5_floor_marked.png", "Third floor hallway with hole in cell #5 floor marked"),
                    _img("house-arrest-part-2", "second_floor_hallway_with_debris_passage_marked.png", "Second floor hallway with debris passage marked"),
                    _img("house-arrest-part-2", "the_door_to_the_warden_s_apartment.png", "The door to the warden's apartment"),
                ],
            },
        },
    },
    "63966faeea19ac7ed845db2c": {
        "normalizedName": "information-source",
        "images": [],
        "objectives": {
            "6397a6942e519e69d2139b25": {
                "type": "mark",
                "description": "Locate and mark the first Patrol-A armored vehicle with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("information-source", "patrol-a_2.png"),
                ],
            },
            "6397a7ce706b793c7d6094c9": {
                "type": "mark",
                "description": "Locate and mark the second Patrol-A armored vehicle with an MS2000 Marker on Streets of Tarkov",
                "images": [
                    _img("information-source", "patrol-a_1.png"),
                ],
            },
        },
    },
    "63966fbeea19ac7ed845db2e": {
        "normalizedName": "missing-informant",
        "images": [],
        "objectives": {
            "63a7cee004d3dc28a52a20f8": {
                "type": "visit",
                "description": "Locate the informant's backup hideout on Streets of Tarkov",
                "images": [
                    _img("missing-informant", "door_to_the_hideout.png", "Door to the hideout"),
                    _img("missing-informant", "first_journal_spawn_tucked_under_the_jacket_pile_on_the_ground_to_the_right_after_entering.png", "First journal spawn: Tucked under the jacket pile on the ground to the right after entering"),
                    _img("missing-informant", "second_journal_spawn_next_to_a_package_in_the_back_right_under_the_rubble.png", "Second journal spawn: Next to a package in the back right, under the rubble"),
                    _img("missing-informant", "third_journal_spawn_tucked_under_a_blue_trash_bag_on_the_ground_next_to_the_shelf_in_the_back_left.png", "Third journal spawn: Tucked under a blue trash bag on the ground next to the shelf in the back left"),
                ],
            },
            "63990a1ddcdd126fef1771c1": {
                "type": "findQuestItem",
                "description": "Obtain the informant's journal",
                "images": [
                    _img("missing-informant", "door_to_the_hideout.png", "Door to the hideout"),
                    _img("missing-informant", "first_journal_spawn_tucked_under_the_jacket_pile_on_the_ground_to_the_right_after_entering.png", "First journal spawn: Tucked under the jacket pile on the ground to the right after entering"),
                    _img("missing-informant", "second_journal_spawn_next_to_a_package_in_the_back_right_under_the_rubble.png", "Second journal spawn: Next to a package in the back right, under the rubble"),
                    _img("missing-informant", "third_journal_spawn_tucked_under_a_blue_trash_bag_on_the_ground_next_to_the_shelf_in_the_back_left.png", "Third journal spawn: Tucked under a blue trash bag on the ground next to the shelf in the back left"),
                ],
            },
        },
    },
    "6396700fea19ac7ed845db32": {
        "normalizedName": "spotter",
        "images": [],
        "objectives": {
            "63a7d2acee7b4d0d5507baf2": {
                "type": "visit",
                "description": "Locate a good sniping position in Concordia overlooking the construction site",
                "images": [
                    _img("spotter", "the_concordia_building.png", "The Concordia building"),
                    _img("spotter", "way_to_apartment_8_from_outside_via_house_no._1_staircase.png", "Way to apartment 8 from outside via house no. 1 staircase"),
                    _img("spotter", "staircase_to_apartment_8.png", "Staircase to apartment 8"),
                    _img("spotter", "hallway_to_apartment_8.png", "Hallway to apartment 8"),
                    _img("spotter", "door_to_apartment_8.png", "Door to apartment 8"),
                    _img("spotter", "red_table_in_the_kitchen.png", "Red table in the kitchen"),
                    _img("spotter", "the_table.png", "The table"),
                ],
            },
            "639c359f0a20a7001f00c7a8": {
                "type": "plantItem",
                "description": "Hide AI Tactical Sound Moderator .338 LM near the red table",
                "images": [
                    _img("spotter", "the_concordia_building.png", "The Concordia building"),
                    _img("spotter", "way_to_apartment_8_from_outside_via_house_no._1_staircase.png", "Way to apartment 8 from outside via house no. 1 staircase"),
                    _img("spotter", "staircase_to_apartment_8.png", "Staircase to apartment 8"),
                    _img("spotter", "hallway_to_apartment_8.png", "Hallway to apartment 8"),
                    _img("spotter", "door_to_apartment_8.png", "Door to apartment 8"),
                    _img("spotter", "red_table_in_the_kitchen.png", "Red table in the kitchen"),
                    _img("spotter", "the_table.png", "The table"),
                ],
            },
            "63a7d315f32fa1316250c3d2": {
                "type": "visit",
                "description": "Locate a good sniping position at Primorsky overlooking the movie theater",
                "images": [
                    _img("spotter", "apartment_10_from_outside.png", "Apartment 10 from outside"),
                    _img("spotter", "the_skybridge_from_the_ground.png", "The skybridge from the ground"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_1.png", "Way to the Primorsky 46 staircase 1"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_2.png", "Way to the Primorsky 46 staircase 2"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_3.png", "Way to the Primorsky 46 staircase 3"),
                    _img("spotter", "primorsky_46_staircase.png", "Primorsky 46 staircase"),
                    _img("spotter", "door_to_the_skybridge_on_the_top_floor.png", "Door to the skybridge on the top floor"),
                    _img("spotter", "the_skybridge_with_the_unlockable_door.png", "The skybridge with the unlockable door"),
                    _img("spotter", "door_to_the_primorsky_48_apartment_10.png", "Door to the Primorsky 48 apartment 10"),
                    _img("spotter", "way_to_the_bedroom.png", "Way to the bedroom"),
                    _img("spotter", "the_bed.png", "The bed"),
                ],
            },
            "639c39807c1532d85b0162a9": {
                "type": "plantItem",
                "description": "Hide Trijicon REAP-IR thermal scope under Makhors' bed",
                "images": [
                    _img("spotter", "apartment_10_from_outside.png", "Apartment 10 from outside"),
                    _img("spotter", "the_skybridge_from_the_ground.png", "The skybridge from the ground"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_1.png", "Way to the Primorsky 46 staircase 1"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_2.png", "Way to the Primorsky 46 staircase 2"),
                    _img("spotter", "way_to_the_primorsky_46_staircase_3.png", "Way to the Primorsky 46 staircase 3"),
                    _img("spotter", "primorsky_46_staircase.png", "Primorsky 46 staircase"),
                    _img("spotter", "door_to_the_skybridge_on_the_top_floor.png", "Door to the skybridge on the top floor"),
                    _img("spotter", "the_skybridge_with_the_unlockable_door.png", "The skybridge with the unlockable door"),
                    _img("spotter", "door_to_the_primorsky_48_apartment_10.png", "Door to the Primorsky 48 apartment 10"),
                    _img("spotter", "way_to_the_bedroom.png", "Way to the bedroom"),
                    _img("spotter", "the_bed.png", "The bed"),
                ],
            },
        },
    },
    "63967028c4a91c5cb76abd81": {
        "normalizedName": "trouble-in-the-big-city",
        "images": [],
        "objectives": {
            "639aea2228d8a21b593a3491": {
                "type": "mark",
                "description": "Mark the cargo area with an MS2000 Marker",
                "images": [
                    _img("trouble-in-the-big-city", "cargo_truck_trailer_with_grate_door.png", "Cargo truck trailer with grate door"),
                    _img("trouble-in-the-big-city", "marker_spot.png", "Marker spot"),
                ],
            },
        },
    },
    "639135534b15ca31f76bc317": {
        "normalizedName": "your-car-needs-a-service",
        "images": [],
        "objectives": {
            "639135534b15ca31f76bc318": {
                "type": "findQuestItem",
                "description": "Obtain the digital storage device from the car dealership manager's office on Streets of Tarkov",
                "images": [
                    _img("your-car-needs-a-service", "the_car_dealership.png", "The car dealership"),
                    _img("your-car-needs-a-service", "grate_door_to_the_closed_section.png", "Grate door to the closed section"),
                    _img("your-car-needs-a-service", "door_to_the_office.png", "Door to the office"),
                    _img("your-car-needs-a-service", "inside_the_office.png", "Inside the office"),
                    _img("your-car-needs-a-service", "flash_drive_on_the_desk.png", "Flash drive on the desk"),
                ],
            },
        },
    },
    "697a299d2b5bc4514201b8a3": {
        "normalizedName": "duck-hunt",
        "images": [],
        "objectives": {
            "duck-hunt-customs01": {
                "type": "shoot",
                "description": "In a broken window on the south side of warehouse 17.",
                "images": [_img("duck-hunt", "duck-hunt-customs01.png", "In a broken window on the south side of warehouse 17.")],
            },
            "duck-hunt-customs02": {
                "type": "shoot",
                "description": "On the roof of the military checkpoint gate.",
                "images": [_img("duck-hunt", "duck-hunt-customs02.png", "On the roof of the military checkpoint gate.")],
            },
            "duck-hunt-customs03": {
                "type": "shoot",
                "description": "On a plank in the electricity pylon southwest of the new gas station.",
                "images": [_img("duck-hunt", "duck-hunt-customs03.png", "On a plank in the electricity pylon southwest of the new gas station.")],
            },
            "duck-hunt-customs04": {
                "type": "shoot",
                "description": "On the corner of the metal scaffolding of the industrial plant.",
                "images": [_img("duck-hunt", "duck-hunt-customs04.png", "On the corner of the metal scaffolding of the industrial plant.")],
            },
            "duck-hunt-customs05": {
                "type": "shoot",
                "description": "On the roof of the three-story dorms building.",
                "images": [_img("duck-hunt", "duck-hunt-customs05.png", "On the roof of the three-story dorms building.")],
            },
            "duck-hunt-customs06": {
                "type": "shoot",
                "description": "In the driver's seat of the tanker truck.",
                "images": [_img("duck-hunt", "duck-hunt-customs06.png", "In the driver's seat of the tanker truck.")],
            },
            "duck-hunt-customs07": {
                "type": "shoot",
                "description": "In a half-open room with a dead Scav on the ground floor.",
                "images": [_img("duck-hunt", "duck-hunt-customs07.png", "In a half-open room with a dead Scav on the ground floor.")],
            },
            "duck-hunt-customs08": {
                "type": "shoot",
                "description": "Next to the center pylon under the bridge.",
                "images": [_img("duck-hunt", "duck-hunt-customs08.png", "Next to the center pylon under the bridge.")],
            },
            "duck-hunt-customs09": {
                "type": "shoot",
                "description": "On top of the administrative office.",
                "images": [_img("duck-hunt", "duck-hunt-customs09.png", "On top of the administrative office.")],
            },
            "duck-hunt-customs10": {
                "type": "shoot",
                "description": "On the roof of the garages.",
                "images": [_img("duck-hunt", "duck-hunt-customs10.png", "On the roof of the garages.")],
            },
            "duck-hunt-shoreline01": {
                "type": "shoot",
                "description": "On top of the bus next to the Terminal Intercom.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline01.png", "On top of the bus next to the Terminal Intercom.")],
            },
            "duck-hunt-shoreline02": {
                "type": "shoot",
                "description": "On top of the weather radar.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline02.png", "On top of the weather radar.")],
            },
            "duck-hunt-shoreline03": {
                "type": "shoot",
                "description": "On top of the lighthouse extraction.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline03.png", "On top of the lighthouse extraction.")],
            },
            "duck-hunt-shoreline04": {
                "type": "shoot",
                "description": "Behind the house on the scav island.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline04.png", "Behind the house on the scav island.")],
            },
            "duck-hunt-shoreline05": {
                "type": "shoot",
                "description": "On a ventilation outside of the cottage building.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline05.png", "On a ventilation outside of the cottage building.")],
            },
            "duck-hunt-shoreline06": {
                "type": "shoot",
                "description": "On top of the shop building in the village.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline06.png", "On top of the shop building in the village.")],
            },
            "duck-hunt-shoreline07": {
                "type": "shoot",
                "description": "Inside the church on the top of the bell tower.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline07.png", "Inside the church on the top of the bell tower.")],
            },
            "duck-hunt-shoreline08": {
                "type": "shoot",
                "description": "Inside the bunker next to climber's trail extraction.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline08.png", "Inside the bunker next to climber's trail extraction.")],
            },
            "duck-hunt-shoreline09": {
                "type": "shoot",
                "description": "On top of the admin building.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline09.png", "On top of the admin building.")],
            },
            "duck-hunt-shoreline10": {
                "type": "shoot",
                "description": "On top of the green container inside the north building at smuggler depot.",
                "images": [_img("duck-hunt", "duck-hunt-shoreline10.png", "On top of the green container inside the north building at smuggler depot.")],
            },
            "duck-hunt-factory01": {
                "type": "shoot",
                "description": "Next to transit to customs on the roof of the north building.",
                "images": [_img("duck-hunt", "duck-hunt-factory01.png", "Next to transit to customs on the roof of the north building.")],
            },
            "duck-hunt-factory02": {
                "type": "shoot",
                "description": "On barrels in the underground.",
                "images": [_img("duck-hunt", "duck-hunt-factory02.png", "On barrels in the underground.")],
            },
            "duck-hunt-factory03": {
                "type": "shoot",
                "description": "Behind the transit to labs fence.",
                "images": [_img("duck-hunt", "duck-hunt-factory03.png", "Behind the transit to labs fence.")],
            },
            "duck-hunt-factory04": {
                "type": "shoot",
                "description": "Showing under the pipe.",
                "images": [_img("duck-hunt", "duck-hunt-factory04.png", "Showing under the pipe.")],
            },
            "duck-hunt-factory05": {
                "type": "shoot",
                "description": "Inside a sink next to the dead corpse.",
                "images": [_img("duck-hunt", "duck-hunt-factory05.png", "Inside a sink next to the dead corpse.")],
            },
            "duck-hunt-factory06": {
                "type": "shoot",
                "description": "Outside the office windows on the roof.",
                "images": [_img("duck-hunt", "duck-hunt-factory06.png", "Outside the office windows on the roof.")],
            },
            "duck-hunt-factory07": {
                "type": "shoot",
                "description": "On top of a beam.",
                "images": [_img("duck-hunt", "duck-hunt-factory07.png", "On top of a beam.")],
            },
            "duck-hunt-factory08": {
                "type": "shoot",
                "description": "On a metallic hook in front of Gate 3 entrance.",
                "images": [_img("duck-hunt", "duck-hunt-factory08.png", "On a metallic hook in front of Gate 3 entrance.")],
            },
            "duck-hunt-factory09": {
                "type": "shoot",
                "description": "On a beam at Gate 3 extraction.",
                "images": [_img("duck-hunt", "duck-hunt-factory09.png", "On a beam at Gate 3 extraction.")],
            },
            "duck-hunt-woods01": {
                "type": "shoot",
                "description": "Shoot down toy duck #1",
                "images": [_img("duck-hunt", "duck-hunt-woods01.png", "Shoot down toy duck #1")],
            },
            "duck-hunt-woods02": {
                "type": "shoot",
                "description": "Shoot down toy duck #2",
                "images": [_img("duck-hunt", "duck-hunt-woods02.png", "Shoot down toy duck #2")],
            },
            "duck-hunt-woods03": {
                "type": "shoot",
                "description": "Shoot down toy duck #3",
                "images": [_img("duck-hunt", "duck-hunt-woods03.png", "Shoot down toy duck #3")],
            },
            "duck-hunt-woods04": {
                "type": "shoot",
                "description": "Shoot down toy duck #4",
                "images": [_img("duck-hunt", "duck-hunt-woods04.png", "Shoot down toy duck #4")],
            },
            "duck-hunt-woods05": {
                "type": "shoot",
                "description": "Shoot down toy duck #5",
                "images": [_img("duck-hunt", "duck-hunt-woods05.png", "Shoot down toy duck #5")],
            },
            "duck-hunt-woods06": {
                "type": "shoot",
                "description": "Shoot down toy duck #6",
                "images": [_img("duck-hunt", "duck-hunt-woods06.png", "Shoot down toy duck #6")],
            },
            "duck-hunt-woods07": {
                "type": "shoot",
                "description": "Shoot down toy duck #7",
                "images": [_img("duck-hunt", "duck-hunt-woods07.png", "Shoot down toy duck #7")],
            },
            "duck-hunt-woods08": {
                "type": "shoot",
                "description": "Shoot down toy duck #8",
                "images": [_img("duck-hunt", "duck-hunt-woods08.png", "Shoot down toy duck #8")],
            },
            "duck-hunt-woods09": {
                "type": "shoot",
                "description": "Shoot down toy duck #9",
                "images": [_img("duck-hunt", "duck-hunt-woods09.png", "Shoot down toy duck #9")],
            },
            "duck-hunt-woods10": {
                "type": "shoot",
                "description": "Shoot down toy duck #10",
                "images": [_img("duck-hunt", "duck-hunt-woods10.png", "Shoot down toy duck #10")],
            },
            "duck-hunt-ground-zero01": {
                "type": "shoot",
                "description": "At the bottom of the nakatani basement stairs, on a box.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero01.png", "At the bottom of the nakatani basement stairs, on a box.")],
            },
            "duck-hunt-ground-zero02": {
                "type": "shoot",
                "description": "In front of the capital insight building, in the tunnel going down.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero02.png", "In front of the capital insight building, in the tunnel going down.")],
            },
            "duck-hunt-ground-zero03": {
                "type": "shoot",
                "description": "On top of the building between capital insight building and Fusion building.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero03.png", "On top of the building between capital insight building and Fusion building.")],
            },
            "duck-hunt-ground-zero04": {
                "type": "shoot",
                "description": "On top of Fusion building on sandbags, in front of Empire building.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero04.png", "On top of Fusion building on sandbags, in front of Empire building.")],
            },
            "duck-hunt-ground-zero05": {
                "type": "shoot",
                "description": "On top of the ceiling above the park in front of the TerraGroup Building.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero05.png", "On top of the ceiling above the park in front of the TerraGroup Building.")],
            },
            "duck-hunt-ground-zero06": {
                "type": "shoot",
                "description": "On the car in the wall above the TerraGroup parking entrance.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero06.png", "On the car in the wall above the TerraGroup parking entrance.")],
            },
            "duck-hunt-ground-zero07": {
                "type": "shoot",
                "description": "In the underground parking, behind the fence on top of a container.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero07.png", "In the underground parking, behind the fence on top of a container.")],
            },
            "duck-hunt-ground-zero08": {
                "type": "shoot",
                "description": "In the underground parking, behind the fence on the ground.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero08.png", "In the underground parking, behind the fence on the ground.")],
            },
            "duck-hunt-ground-zero09": {
                "type": "shoot",
                "description": "On top of the Tartower sales office exit building.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero09.png", "On top of the Tartower sales office exit building.")],
            },
            "duck-hunt-ground-zero10": {
                "type": "shoot",
                "description": "Inside the Mira avenue flare exit, inside a baby stroller on the left side of the road.",
                "images": [_img("duck-hunt", "duck-hunt-ground-zero10.png", "Inside the Mira avenue flare exit, inside a baby stroller on the left side of the road.")],
            },
            "duck-hunt-interchange01": {
                "type": "shoot",
                "description": "On top of a container on the north-east unfinished road.",
                "images": [_img("duck-hunt", "duck-hunt-interchange01.png", "On top of a container on the north-east unfinished road.")],
            },
            "duck-hunt-interchange02": {
                "type": "shoot",
                "description": "On the staircase behind IDEA.",
                "images": [_img("duck-hunt", "duck-hunt-interchange02.png", "On the staircase behind IDEA.")],
            },
            "duck-hunt-interchange03": {
                "type": "shoot",
                "description": "Above the top boxes on a storage shed behind IDEA.",
                "images": [_img("duck-hunt", "duck-hunt-interchange03.png", "Above the top boxes on a storage shed behind IDEA.")],
            },
            "duck-hunt-interchange04": {
                "type": "shoot",
                "description": "Above a storage shed next to goshan storage entrance behind OLI.",
                "images": [_img("duck-hunt", "duck-hunt-interchange04.png", "Above a storage shed next to goshan storage entrance behind OLI.")],
            },
            "duck-hunt-interchange05": {
                "type": "shoot",
                "description": "Behind the green container behind OLI.",
                "images": [_img("duck-hunt", "duck-hunt-interchange05.png", "Behind the green container behind OLI.")],
            },
            "duck-hunt-interchange06": {
                "type": "shoot",
                "description": "To the ceiling above the Goshan fire.",
                "images": [_img("duck-hunt", "duck-hunt-interchange06.png", "To the ceiling above the Goshan fire.")],
            },
            "duck-hunt-interchange07": {
                "type": "shoot",
                "description": "Behind Scav Camp Co-Op extract.",
                "images": [_img("duck-hunt", "duck-hunt-interchange07.png", "Behind Scav Camp Co-Op extract.")],
            },
            "duck-hunt-interchange08": {
                "type": "shoot",
                "description": "Behind custom transit area.",
                "images": [_img("duck-hunt", "duck-hunt-interchange08.png", "Behind custom transit area.")],
            },
            "duck-hunt-interchange09": {
                "type": "shoot",
                "description": "On top of a container loader.",
                "images": [_img("duck-hunt", "duck-hunt-interchange09.png", "On top of a container loader.")],
            },
            "duck-hunt-interchange10": {
                "type": "shoot",
                "description": "On the back of the truck at Path to River flare extraction.",
                "images": [_img("duck-hunt", "duck-hunt-interchange10.png", "On the back of the truck at Path to River flare extraction.")],
            },
            "duck-hunt-reserve01": {
                "type": "shoot",
                "description": "Behind Scav Lands extraction on a dumpster.",
                "images": [_img("duck-hunt", "duck-hunt-reserve01.png", "Behind Scav Lands extraction on a dumpster.")],
            },
            "duck-hunt-reserve02": {
                "type": "shoot",
                "description": "Behind secret extraction Exit to Woods on a LAV body.",
                "images": [_img("duck-hunt", "duck-hunt-reserve02.png", "Behind secret extraction Exit to Woods on a LAV body.")],
            },
            "duck-hunt-reserve03": {
                "type": "shoot",
                "description": "To the ceiling inside of the Locomotive repair depot.",
                "images": [_img("duck-hunt", "duck-hunt-reserve03.png", "To the ceiling inside of the Locomotive repair depot.")],
            },
            "duck-hunt-reserve04": {
                "type": "shoot",
                "description": "Inside the truck cabin by the the south-west tower.",
                "images": [_img("duck-hunt", "duck-hunt-reserve04.png", "Inside the truck cabin by the the south-west tower.")],
            },
            "duck-hunt-reserve05": {
                "type": "shoot",
                "description": "On top of the repair center.",
                "images": [_img("duck-hunt", "duck-hunt-reserve05.png", "On top of the repair center.")],
            },
            "duck-hunt-reserve06": {
                "type": "shoot",
                "description": "Inside the ceiling of the parking at south checkpoint.",
                "images": [_img("duck-hunt", "duck-hunt-reserve06.png", "Inside the ceiling of the parking at south checkpoint.")],
            },
            "duck-hunt-reserve07": {
                "type": "shoot",
                "description": "South of the dome on top of the highest rock.",
                "images": [_img("duck-hunt", "duck-hunt-reserve07.png", "South of the dome on top of the highest rock.")],
            },
            "duck-hunt-reserve08": {
                "type": "shoot",
                "description": "On top of a rock east of black pawn building.",
                "images": [_img("duck-hunt", "duck-hunt-reserve08.png", "On top of a rock east of black pawn building.")],
            },
            "duck-hunt-reserve09": {
                "type": "shoot",
                "description": "Inside the ventilation shaft of the helicopter on the east side.",
                "images": [_img("duck-hunt", "duck-hunt-reserve09.png", "Inside the ventilation shaft of the helicopter on the east side.")],
            },
            "duck-hunt-reserve10": {
                "type": "shoot",
                "description": "Inside the hole above the door from bunker to D-2 exit.",
                "images": [_img("duck-hunt", "duck-hunt-reserve10.png", "Inside the hole above the door from bunker to D-2 exit.")],
            },
        },
    },
}
# Total quests with images: 44
# Total objectives with images: 180


def get_objective_images(quest_id: str, objective_id: str) -> list:
    """Get images for a specific objective. Returns [] if none."""
    quest = QUEST_SCREENSHOTS.get(quest_id)
    if not quest:
        return []
    obj = quest.get("objectives", {}).get(objective_id)
    if obj and obj.get("images"):
        return obj["images"]
    return quest.get("images", [])


def get_quest_images(quest_id: str) -> list:
    """Get quest-level images (fallback for any objective). Returns [] if none."""
    quest = QUEST_SCREENSHOTS.get(quest_id)
    return quest.get("images", []) if quest else []
