-- Quest Tracker Database Schema
-- Version: 3
-- Database: quest_tracker_{mode}.db
-- Managed by: DatabaseManager (PySide6)
--
-- This schema defines the structure for the quest tracker database,
-- which stores quest states, objectives, item progress, and migration history.

-- ============================================================================
-- TABLE: quest_definitions
-- Purpose: Quest metadata with version tracking using MD5 hashes
-- Used to detect quest definition changes from game updates
-- ============================================================================
CREATE TABLE IF NOT EXISTS quest_definitions (
    quest_id TEXT PRIMARY KEY,
    quest_name TEXT,
    definition_hash TEXT,
    items_hash TEXT,
    objectives_hash TEXT,
    prerequisites_hash TEXT,
    last_seen_version TEXT,
    first_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_deprecated BOOLEAN DEFAULT 0
);

-- ============================================================================
-- TABLE: favorite_quests
-- Purpose: User-favorited quests for quick access
-- ============================================================================
CREATE TABLE IF NOT EXISTS favorite_quests (
    quest_id TEXT PRIMARY KEY,
    quest_name TEXT,
    added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    added_by TEXT
);

-- ============================================================================
-- TABLE: banned_quests
-- Purpose: Quests that user has banned/removed from quest tree
-- ============================================================================
CREATE TABLE IF NOT EXISTS banned_quests (
    quest_id TEXT PRIMARY KEY,
    quest_name TEXT,
    banned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    banned_by TEXT,
    banned_reason TEXT,
    parent_banned_quest_id TEXT
);

-- ============================================================================
-- TABLE: quest_states
-- Purpose: Current completion status of each quest
-- States: locked, ready, active, complete, failed
-- ============================================================================
CREATE TABLE IF NOT EXISTS quest_states (
    quest_id TEXT PRIMARY KEY,
    quest_name TEXT,
    state TEXT NOT NULL,
    completed_at TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_by TEXT,
    definition_hash TEXT,
    is_valid BOOLEAN DEFAULT 1
);

-- ============================================================================
-- TABLE: item_progress
-- Purpose: Item collection progress for "find in raid" quest objectives
-- Example: "Find 5 FIR Gas Analyzers" → found_count=3, required_count=5
-- ============================================================================
CREATE TABLE IF NOT EXISTS item_progress (
    quest_id TEXT,
    item_id TEXT,
    item_name TEXT,
    normalized_name TEXT,
    found_count INTEGER DEFAULT 0,
    required_count INTEGER,
    found_in_raid BOOLEAN DEFAULT 0,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_by TEXT,
    is_valid BOOLEAN DEFAULT 1,
    PRIMARY KEY (quest_id, item_id)
);

-- ============================================================================
-- TABLE: objective_progress
-- Purpose: Objective completion counts for multi-step objectives
-- Example: "Kill 10 PMCs on Customs" → current_count=7, total_count=10
-- ============================================================================
CREATE TABLE IF NOT EXISTS objective_progress (
    quest_id TEXT,
    objective_id TEXT,
    objective_description TEXT,
    objective_type TEXT,
    current_count INTEGER DEFAULT 0,
    total_count INTEGER,
    is_complete BOOLEAN DEFAULT 0,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_by TEXT,
    is_valid BOOLEAN DEFAULT 1,
    PRIMARY KEY (quest_id, objective_id)
);

-- ============================================================================
-- TABLE: choice_item_progress
-- Purpose: Tracks quest reward choices AND ANY objective specific items
-- Example 1 (reward): Choose "3× M855A1" or "2× BT" → stores selection
-- Example 2 (ANY objective): "Hand over ANY 3 FIR medicine" → {AFAK:2, AI-2:1}
-- ============================================================================
CREATE TABLE IF NOT EXISTS choice_item_progress (
    quest_id TEXT,
    choice_id TEXT,
    item_id TEXT,
    item_name TEXT,
    count INTEGER DEFAULT 0,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_by TEXT,
    is_valid BOOLEAN DEFAULT 1,
    PRIMARY KEY (quest_id, choice_id, item_id)
);

-- ============================================================================
-- TABLE: bsg_log_imports
-- Purpose: Audit log for automatic quest state imports from EFT game logs
-- ============================================================================
CREATE TABLE IF NOT EXISTS bsg_log_imports (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    profile_id TEXT,
    mode TEXT,
    import_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_count INTEGER,
    active_count INTEGER,
    cutoff_date TIMESTAMP
);

-- ============================================================================
-- TABLE: version_history
-- Purpose: Tracks quest data version changes for migration history
-- ============================================================================
CREATE TABLE IF NOT EXISTS version_history (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    version_hash TEXT UNIQUE,
    version_data TEXT,
    applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    migration_results TEXT
);

-- ============================================================================
-- TABLE: migration_log
-- Purpose: Detailed log of all quest definition migrations
-- ============================================================================
CREATE TABLE IF NOT EXISTS migration_log (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    quest_id TEXT,
    migration_type TEXT,
    old_data TEXT,
    new_data TEXT,
    migrated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- ============================================================================
-- TABLE: change_log
-- Purpose: Real-time change events for IPC (Inter-Process Communication)
-- Used to synchronize changes between main app, quest tree, and BSG monitor
-- ============================================================================
CREATE TABLE IF NOT EXISTS change_log (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    table_name TEXT NOT NULL,
    record_id TEXT NOT NULL,
    change_type TEXT NOT NULL,  -- INSERT, UPDATE, DELETE
    changed_by TEXT NOT NULL,   -- app_identifier
    change_data TEXT,           -- JSON
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- ============================================================================
-- TABLE: settings
-- Purpose: Application settings in key-value format
-- ============================================================================
CREATE TABLE IF NOT EXISTS settings (
    key TEXT PRIMARY KEY,
    value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_by TEXT
);

-- ============================================================================
-- INDEXES
-- ============================================================================
CREATE INDEX IF NOT EXISTS idx_quest_definitions_hash ON quest_definitions(definition_hash);
CREATE INDEX IF NOT EXISTS idx_quest_states_valid ON quest_states(is_valid);
CREATE INDEX IF NOT EXISTS idx_quest_states_updated ON quest_states(updated_at);
CREATE INDEX IF NOT EXISTS idx_item_progress_valid ON item_progress(is_valid);
CREATE INDEX IF NOT EXISTS idx_objective_progress_valid ON objective_progress(is_valid);
CREATE INDEX IF NOT EXISTS idx_migration_log_quest ON migration_log(quest_id);
CREATE INDEX IF NOT EXISTS idx_banned_quests_reason ON banned_quests(banned_reason);
CREATE INDEX IF NOT EXISTS idx_change_log_timestamp ON change_log(timestamp);
CREATE INDEX IF NOT EXISTS idx_change_log_app ON change_log(changed_by, timestamp);

-- ============================================================================
-- SCHEMA VERSION
-- Store schema version in settings table for migration tracking
-- ============================================================================
INSERT OR IGNORE INTO settings (key, value) VALUES ('schema_version', '3');
